
import xbmcaddon, xbmcvfs, xbmcgui
import re
import json
import os
from threading import Thread
try: from urlparse import urlparse
except ImportError: from urllib.parse import urlparse
from modules.utils import get_release_quality, get_file_info, supported_video_extensions, clean_title, normalize
from modules import settings
# from modules.utils import logger

__addon__ = xbmcaddon.Addon(id='plugin.video.fen')
window = xbmcgui.Window(10000)

class DownloadsSource:
    def __init__(self):
        self.scrape_provider = 'downloads'
        self.threads  = []
        self.sources = []
        self.scrape_results = []
        self.highlight_type = __addon__.getSetting('highlight.type')
        if self.highlight_type == '': self.highlight_type = '0'
        self.provider_color = settings.provider_color(self.scrape_provider)
        self.second_line_color = __addon__.getSetting('secondline.identify')
        if self.second_line_color == 'No Color': self.second_line_color = ''
        self.show_extra_info = settings.show_extra_info()
        self.show_filenames = settings.show_filenames()
        self.extensions = supported_video_extensions()
        self.extensions.remove('.strm')

    def results(self, info):
        try:
            self.info = info
            self.db_type = self.info.get("db_type")
            self.download_path = settings.download_directory(self.db_type)
            self.title = self.info.get("title")
            self.year = self.info.get("year")
            self.season = self.info.get("season")
            self.episode = self.info.get("episode")
            self.title_query = clean_title(self.title)
            self.folder_query = self._season_query_list() if self.db_type == 'episode' else self._year_query_list()
            self.file_query = self._episode_query_list() if self.db_type == 'episode' else self._year_query_list()
            self._scrape_directory(self.download_path)
            if not self.scrape_results: return self.sources
            self._label_settings()
            for item in self.scrape_results:
                file_name = item[0]
                file_dl = item[1]
                size = self._get_size(file_dl)
                details = get_file_info(file_name)
                video_quality = get_release_quality(file_name, file_dl)
                labels = self._build_label(file_name, details, size, video_quality)
                label = labels[0]
                multiline_label = labels[1]
                self.sources.append({'name': file_name,
                                    'label': label,
                                    'multiline_label': multiline_label,
                                    'title': file_name,
                                    'quality': video_quality,
                                    'size': size,
                                    'url_dl': file_dl,
                                    'id': file_dl,
                                    'downloads': True,
                                    'direct': True,
                                    'source': self.scrape_provider,
                                    'scrape_provider': self.scrape_provider})
            
            window.setProperty('downloads_source_results', json.dumps(self.sources))
        except Exception as e:
            from modules.utils import logger
            logger('FEN downloads scraper Exception', e)
        
        return self.sources

    def _scrape_directory(self, folder_name):
        folder_results = []
        dirs, files = xbmcvfs.listdir(folder_name)
        cloud_files = files + dirs
        for item in cloud_files:
            ext = os.path.splitext(urlparse(item).path)[-1]
            file_type = 'file' if ext else 'folder'
            item_name = clean_title(normalize(item))
            if file_type == 'file' and ext in self.extensions:
                if self.db_type == 'movie':
                    if self.title_query in item_name:
                        url_path = self.url_path(folder_name, item)
                        self.scrape_results.append((item, url_path))
                elif any(x in item_name for x in self.file_query):
                    if not folder_name == self.download_path:
                        url_path = self.url_path(folder_name, item)
                        self.scrape_results.append((item, url_path))
                    elif self.title_query in item_name:
                        url_path = self.url_path(folder_name, item)
                        self.scrape_results.append((item, url_path))    
            elif file_type == 'folder':
                if self.title_query in item_name or any(x in item_name for x in self.folder_query):
                    new_folder = os.path.join(folder_name, item)
                    folder_results.append(new_folder)
        if not folder_results: return
        return self._scraper_worker(folder_results)

    def _scraper_worker(self, folder_results):
        threads = []
        for i in folder_results: threads.append(Thread(target=self._scrape_directory, args=(i,)))
        [i.start() for i in threads]
        [i.join() for i in threads]

    def url_path(self, folder, file):
        url_path = os.path.join(folder, file)
        return url_path

    def _get_size(self, file):
        f = xbmcvfs.File(file)
        s = f.size()
        f.close()
        size = float(s)/1073741824
        return size

    def _year_query_list(self):
        return (str(self.year), str(int(self.year)+1), str(int(self.year)-1))

    def _season_query_list(self):
        return ['season%02d' % int(self.season), 'season%s' % self.season]

    def _episode_query_list(self):
        return ['s%02de%02d' % (int(self.season), int(self.episode)),
                '%dx%02d' % (int(self.season), int(self.episode)),
                '%02dx%02d' % (int(self.season), int(self.episode)),
                'season%02depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%d' % (int(self.season), int(self.episode))]

    def _label_settings(self):
        if self.highlight_type == '1':
            self.highlight_4K = __addon__.getSetting('scraper_4k_highlight')
            if self.highlight_4K == '': self.highlight_4K = 'magenta'
            self.highlight_1080p = __addon__.getSetting('scraper_1080p_highlight')
            if self.highlight_1080p == '': self.highlight_1080p = 'lawngreen'
            self.highlight_720p = __addon__.getSetting('scraper_720p_highlight')
            if self.highlight_720p == '': self.highlight_720p = 'gold'
            self.highlight_SD = __addon__.getSetting('scraper_SD_highlight')
            if self.highlight_SD == '': self.highlight_SD = 'lightsaltegray'
        else:
            # Single Line...
            if self.provider_color == '':
                self.single_leading = ''
                self.single_closing = ''
            else:
                self.single_leading = '[COLOR=%s]' % self.provider_color
                self.single_closing = '[/COLOR]'
            # Multiline...
            if self.provider_color == '':
                self.multi1_leading = ''
                self.multi1_closing = ''
            else:
                self.multi1_leading = '[COLOR=%s]' % self.provider_color
                self.multi1_closing = '[/COLOR]'
            if self.second_line_color == '':
                self.multi2_leading = self.multi1_leading
                self.multi2_closing = self.multi1_closing
            else:
                self.multi2_leading = '[COLOR=%s]' % self.second_line_color
                self.multi2_closing = '[/COLOR]'

    def _build_label(self, file_name, details, size, video_quality):
        if self.show_filenames: filename = file_name.replace('.', ' ')
        else: filename = ''
        if self.show_extra_info: details = details
        else: details = ''
        label = '[B]DOWN[/B] | [I][B]%s[/B][/I] | %.2f GB' % (video_quality, size)
        multiline_label1 = '[B]DOWN[/B] | [I][B]%s[/B][/I] | %.2f GB' % (video_quality, size)
        if self.show_extra_info: label += ' | %s' % details
        multiline_label2 = ''
        if self.show_filenames:
            label += ' | %s' % filename
            multiline_label1 += ' | %s' % details
            multiline_label2 += '\n        %s' % filename
        else:
            multiline_label2 += '\n        %s' % details
        label = label.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
        label = re.sub('\[I\]\s+\[/I\]', ' ', label)
        label = re.sub('\|\s+\|', '|', label)
        label = re.sub('\|(?:\s+|)$', '', label)
        label = label.upper()
        multiline_label1 = multiline_label1.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
        multiline_label1 = re.sub('\[I\]\s+\[/I\]', ' ', multiline_label1)
        multiline_label1 = re.sub('\|\s+\|', '|', multiline_label1)
        multiline_label1 = re.sub('\|(?:\s+|)$', '', multiline_label1)
        multiline_label1 = multiline_label1.upper()
        if multiline_label2 != '':
            multiline_label2 = multiline_label2.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
            multiline_label2 = re.sub('\[I\]\s+\[/I\]', ' ', multiline_label2)
            multiline_label2 = re.sub('\|\s+\|', '|', multiline_label2)
            multiline_label2 = re.sub('\|(?:\s+|)$', '', multiline_label2)
            multiline_label2 = multiline_label2.upper()
        if self.highlight_type == '1':
            if video_quality.upper() == '4K': leading_color = self.highlight_4K
            elif video_quality.upper()  == '1080P': leading_color = self.highlight_1080p
            elif video_quality.upper() == '720P': leading_color = self.highlight_720p
            else: leading_color = self.highlight_SD
            if self.second_line_color == '': multiline_open = leading_color
            else: multiline_open = self.second_line_color
            label = '[COLOR=%s]' % leading_color + label + '[/COLOR]'
            multiline_label = '[COLOR=%s]' % leading_color + multiline_label1 + '[/COLOR]' + '[COLOR=%s]' % multiline_open + multiline_label2 + '[/COLOR]'
        else:
            label = self.single_leading + label + self.single_closing
            multiline_label = self.multi1_leading + multiline_label1 + self.multi1_closing + self.multi2_leading + multiline_label2 + self.multi2_closing
        return label, multiline_label
