
import xbmcaddon, xbmcvfs, xbmcgui
import re
import json
import os
from threading import Thread
from apis.real_debrid_api import RealDebridAPI
from modules import fen_cache
from modules.utils import get_release_quality, get_file_info, clean_title, clean_file_name, normalize, supported_video_extensions
from modules import settings
# from modules.utils import logger

__addon__ = xbmcaddon.Addon(id='plugin.video.fen')
window = xbmcgui.Window(10000)

RealDebrid = RealDebridAPI()
_cache = fen_cache.FenCache()

class RealDebridSource:
    def __init__(self):
        self.scrape_provider = 'rd-cloud'
        self.sources = []
        self.folder_results = []
        self.scrape_results = []
        self.highlight_type = __addon__.getSetting('highlight.type')
        if self.highlight_type == '': self.highlight_type = '0'
        self.provider_color = settings.provider_color(self.scrape_provider)
        self.second_line_color = __addon__.getSetting('secondline.identify')
        if self.second_line_color == 'No Color': self.second_line_color = ''
        self.show_extra_info = settings.show_extra_info()
        self.show_filenames = settings.show_filenames()

    def results(self, info):
        try:
            self.info = info
            self.db_type = self.info.get("db_type")
            self.title = self.info.get("title")
            self.year = self.info.get("year")
            if self.year: self.rootname = '%s (%s)' % (self.title, self.year)
            else: self.rootname = self.title
            self.season = self.info.get("season", None)
            self.episode = self.info.get("episode", None)
            self.extensions = supported_video_extensions()
            self.folder_query = clean_title(normalize(self.title))
            self.query_list = self._year_query_list() if self.db_type == 'movie' else self._episode_query_list()
            self._scrape_cloud()
            self._scrape_history()
            if not self.scrape_results: return self.sources
            self._label_settings()
            for item in self.scrape_results:
                try:
                    self.direct_debrid_link = item.get('direct_debrid_link', False)
                    self.folder_name = item['folder_name']
                    self.file_name = self._get_filename(item['path'])
                    self.file_dl = item['url_link']
                    self.size = float(item['bytes'])/1073741824
                    self.video_quality = get_release_quality(self.file_name, self.folder_name)
                    self.details = get_file_info(self.file_name)
                    if not self.details: self.details = get_file_info(self.folder_name)
                    labels = self._build_label()
                    label = labels[0]
                    multiline_label = labels[1]
                    self.sources.append({'name': self.file_name,
                                        'label': label,
                                        'multiline_label': multiline_label,
                                        'title': self.file_name,
                                        'quality': self.video_quality,
                                        'size': self.size,
                                        'url_dl': self.file_dl,
                                        'id': self.file_dl,
                                        'downloads': False,
                                        'direct': True,
                                        'source': self.scrape_provider,
                                        'scrape_provider': self.scrape_provider,
                                        'direct_debrid_link': self.direct_debrid_link})
                except: pass

            window.setProperty('rd-cloud_source_results', json.dumps(self.sources))
        except Exception as e:
            from modules.utils import logger
            logger('FEN real-debrid scraper Exception', e)
        
        return self.sources

    def _assigned_content(self, raw_name):
        try:
            string = 'FEN_RD_%s' % raw_name
            return _cache.get(string)
        except:
            return False

    def _scrape_cloud(self):
        try:
            try: my_cloud_files = RealDebrid.user_cloud()
            except: return self.sources
            for item in my_cloud_files:
                folder_name = clean_title(normalize(item['filename']))
                assigned_content = self._assigned_content(normalize(item['filename']))
                if assigned_content:
                    if assigned_content == self.rootname:
                        self.folder_results.append((normalize(item['filename']), item['id'], True))
                elif self.folder_query in folder_name or not folder_name:
                    self.folder_results.append((normalize(item['filename']), item['id'], False))
            if not self.folder_results: return self.sources
            threads = []
            for i in self.folder_results: threads.append(Thread(target=self._scrape_folders, args=(i,)))
            [i.start() for i in threads]
            [i.join() for i in threads]
        except: pass

    def _scrape_folders(self, folder_info):
        try:
            assigned_folder = folder_info[2]
            folder_files = RealDebrid.user_cloud_info(folder_info[1])
            file_info = [i for i in folder_files['files'] if i['path'].lower().endswith(tuple(self.extensions))]
            file_urls = folder_files['links']
            for c, i in enumerate(file_info):
                try: i.update({'folder_name': folder_info[0], 'url_link': file_urls[c]})
                except: pass
            contents = sorted(file_info, key=lambda k: k['path'])
            for item in contents:
                filename = clean_title(normalize(item['path']))
                if any(x in filename for x in self.query_list):
                    if assigned_folder:
                        self.scrape_results.append(item)
                    elif self.folder_query in filename:
                        self.scrape_results.append(item)
        except: pass

    def _scrape_history(self):
        try:
            my_downloads = RealDebrid.downloads()
            my_downloads = [i for i in my_downloads if i['download'].lower().endswith(tuple(self.extensions))]
            for item in my_downloads:
                filename = clean_title(normalize(item['filename']))
                if any(x in filename for x in self.query_list):
                    if self.folder_query in filename:
                        item = self.make_history_item(item)
                        if item['path'] not in [d['path'] for d in self.scrape_results]:
                            self.scrape_results.append(item)
        except: pass

    def make_history_item(self, history_item):
        finished_item = {}
        finished_item['folder_name'] = history_item['filename']
        finished_item['url_link'] = history_item['download']
        finished_item['bytes'] = history_item['filesize']
        finished_item['path'] = history_item['filename']
        finished_item['direct_debrid_link'] = True
        return finished_item

    def _year_query_list(self):
        return [str(self.year), str(int(self.year)+1), str(int(self.year)-1)]

    def _episode_query_list(self):
        return ['s%02de%02d' % (int(self.season), int(self.episode)),
                '%dx%02d' % (int(self.season), int(self.episode)),
                '%02dx%02d' % (int(self.season), int(self.episode)),
                'season%02depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%d' % (int(self.season), int(self.episode))]

    def _get_filename(self, name):
        if name.startswith('/'): name = name.split('/')[-1]
        return clean_file_name(name)

    def _label_settings(self):
        if self.highlight_type == '1':
            self.highlight_4K = __addon__.getSetting('scraper_4k_highlight')
            if self.highlight_4K == '': self.highlight_4K = 'magenta'
            self.highlight_1080p = __addon__.getSetting('scraper_1080p_highlight')
            if self.highlight_1080p == '': self.highlight_1080p = 'lawngreen'
            self.highlight_720p = __addon__.getSetting('scraper_720p_highlight')
            if self.highlight_720p == '': self.highlight_720p = 'gold'
            self.highlight_SD = __addon__.getSetting('scraper_SD_highlight')
            if self.highlight_SD == '': self.highlight_SD = 'lightsaltegray'
        else:
            # Single Line...
            if self.provider_color == '':
                self.single_leading = ''
                self.single_closing = ''
            else:
                self.single_leading = '[COLOR=%s]' % self.provider_color
                self.single_closing = '[/COLOR]'
            # Multiline...
            if self.provider_color == '':
                self.multi1_leading = ''
                self.multi1_closing = ''
            else:
                self.multi1_leading = '[COLOR=%s]' % self.provider_color
                self.multi1_closing = '[/COLOR]'
            if self.second_line_color == '':
                self.multi2_leading = self.multi1_leading
                self.multi2_closing = self.multi1_closing
            else:
                self.multi2_leading = '[COLOR=%s]' % self.second_line_color
                self.multi2_closing = '[/COLOR]'

    def _build_label(self):
        if self.show_filenames: filename = self.file_name.replace('.', ' ')
        else: filename = ''
        if self.show_extra_info: details = self.details
        else: details = ''
        label = '[B]RD CLOUD[/B] | [I][B]%s[/B][/I] | %.2f GB' % (self.video_quality, self.size)
        multiline_label1 = '[B]RD CLOUD[/B] | [I][B]%s[/B][/I] | %.2f GB' % (self.video_quality, self.size)
        if self.show_extra_info: label += ' | %s' % details
        multiline_label2 = ''
        if self.show_filenames:
            label += ' | %s' % filename
            multiline_label1 += ' | %s' % details
            multiline_label2 += '\n        %s' % filename
        else:
            multiline_label2 += '\n        %s' % details
        label = label.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
        label = re.sub('\[I\]\s+\[/I\]', ' ', label)
        label = re.sub('\|\s+\|', '|', label)
        label = re.sub('\|(?:\s+|)$', '', label)
        label = label.upper()
        multiline_label1 = multiline_label1.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
        multiline_label1 = re.sub('\[I\]\s+\[/I\]', ' ', multiline_label1)
        multiline_label1 = re.sub('\|\s+\|', '|', multiline_label1)
        multiline_label1 = re.sub('\|(?:\s+|)$', '', multiline_label1)
        multiline_label1 = multiline_label1.upper()
        if multiline_label2 != '':
            multiline_label2 = multiline_label2.replace('| 0 |', '|').replace(' | [I]0 [/I]', '')
            multiline_label2 = re.sub('\[I\]\s+\[/I\]', ' ', multiline_label2)
            multiline_label2 = re.sub('\|\s+\|', '|', multiline_label2)
            multiline_label2 = re.sub('\|(?:\s+|)$', '', multiline_label2)
            multiline_label2 = multiline_label2.upper()
        if self.highlight_type == '1':
            if self.video_quality.upper() == '4K': leading_color = self.highlight_4K
            elif self.video_quality.upper()  == '1080P': leading_color = self.highlight_1080p
            elif self.video_quality.upper() == '720P': leading_color = self.highlight_720p
            else: leading_color = self.highlight_SD
            if self.second_line_color == '': multiline_open = leading_color
            else: multiline_open = self.second_line_color
            label = '[COLOR=%s]' % leading_color + label + '[/COLOR]'
            multiline_label = '[COLOR=%s]' % leading_color + multiline_label1 + '[/COLOR]' + '[COLOR=%s]' % multiline_open + multiline_label2 + '[/COLOR]'
        else:
            label = self.single_leading + label + self.single_closing
            multiline_label = self.multi1_leading + multiline_label1 + self.multi1_closing + self.multi2_leading + multiline_label2 + self.multi2_closing
        return label, multiline_label


