
import xbmcaddon, xbmcvfs, xbmcgui
import re
import json
import os
from threading import Thread
from apis.premiumize_api import PremiumizeAPI
from modules import fen_cache
from modules.utils import get_release_quality, get_file_info, clean_title, clean_file_name, normalize, supported_video_extensions
from scrapers import build_internal_scrapers_label
from modules import settings
# from modules.utils import logger

__addon__ = xbmcaddon.Addon(id='plugin.video.fen')
window = xbmcgui.Window(10000)

Premiumize = PremiumizeAPI()
_cache = fen_cache.FenCache()

class PremiumizeSource:
    def __init__(self):
        self.scrape_provider = 'pm-cloud'
        self.sources = []
        self.threads  = []
        self.scrape_results = []

    def results(self, info):
        try:
            self.info = info
            self.db_type = self.info.get("db_type")
            self.title = self.info.get("title")
            self.year = self.info.get("year")
            if self.year: self.rootname = '%s (%s)' % (self.title, self.year)
            else: self.rootname = self.title
            self.season = self.info.get("season")
            self.episode = self.info.get("episode")
            self.folder_query = clean_title(self.title)
            self.season_folder_query = self._season_query_list() if self.db_type == 'episode' else self._year_query_list()
            self.file_query = self._episode_query_list() if self.db_type == 'episode' else self._year_query_list()
            self.extensions = supported_video_extensions()
            self._scrape_cloud()
            if not self.scrape_results: return self.sources
            self._label_settings()
            for item in self.scrape_results:
                try:
                    folder_name = item['folder_name']
                    file_name = clean_file_name(item['name'])
                    file_dl = item['link']
                    size = float(item['size'])/1073741824
                    video_quality = get_release_quality(file_name, folder_name)
                    details = get_file_info(file_name)
                    if not details: details = get_file_info(folder_name)
                    labels = self._build_label(file_name, details, size, video_quality)
                    label = labels[0]
                    multiline_label = labels[1]
                    self.sources.append({'name': file_name,
                                        'label': label,
                                        'multiline_label': multiline_label,
                                        'title': file_name,
                                        'quality': video_quality,
                                        'size': size,
                                        'url_dl': file_dl,
                                        'id': file_dl,
                                        'downloads': False,
                                        'direct': True,
                                        'source': self.scrape_provider,
                                        'scrape_provider': self.scrape_provider})
                except: pass

            window.setProperty('pm-cloud_source_results', json.dumps(self.sources))
        except Exception as e:
            from modules.utils import logger
            logger('FEN premiumize scraper Exception', e)
        
        return self.sources

    def _assigned_content(self, raw_name):
        try:
            string = 'FEN_PM_%s' % raw_name
            return _cache.get(string)
        except:
            return False

    def _scrape_cloud(self, folder_id=None, folder_name=None, assigned_folder=False):
        folder_results = []
        try:
            cloud_files = Premiumize.user_cloud(folder_id)['content']
            cloud_files = [i for i in cloud_files if ('link' in i and i['link'].lower().endswith(tuple(self.extensions))) or i['type'] == 'folder']
            cloud_files = sorted(cloud_files, key=lambda k: k['name'])
            cloud_files = sorted(cloud_files, key=lambda k: k['type'], reverse=True)
        except: return self.sources
        for item in cloud_files:
            file_type = item['type']
            item_name = clean_title(normalize(item['name']))
            if file_type == 'folder':
                if not folder_name:
                    assigned_content = self._assigned_content(normalize(item['name']))
                    if assigned_content:
                        if assigned_content == self.rootname:
                            item['folder_name'] = item['name']
                            item['assigned_content'] = True
                            folder_results.append(item)
                    elif self.folder_query in item_name or any(x in item_name for x in self.season_folder_query):
                        item['folder_name'] = item['name']
                        folder_results.append(item)
                elif self.folder_query in item_name or any(x in item_name for x in self.season_folder_query):
                    item['folder_name'] = item['name']
                    if assigned_folder: item['assigned_content'] = True
                    folder_results.append(item)
            else:
                if assigned_folder:
                    if any(x in item_name for x in self.file_query):
                        item['folder_name'] = folder_name
                        self.scrape_results.append(item)
                elif self.folder_query in item_name and any(x in item_name for x in self.file_query):
                    item['folder_name'] = folder_name
                    self.scrape_results.append(item)
        if not folder_results:
            return
        return self._pm_worker(folder_results)

    def _pm_worker(self, folder_results):
        threads = []
        for i in folder_results: threads.append(Thread(target=self._scrape_cloud, args=(i.get('id'), i.get('folder_name', None), i.get('assigned_content', False))))
        [i.start() for i in threads]
        [i.join() for i in threads]

    def _year_query_list(self):
        return [str(self.year), str(int(self.year)+1), str(int(self.year)-1)]

    def _season_query_list(self):
        return ['s%02d' % int(self.season), 's%d' % int(self.season),
                'season%02d' % int(self.season), 'season%d' % int(self.season)]

    def _episode_query_list(self):
        return ['s%02de%02d' % (int(self.season), int(self.episode)),
                '%dx%02d' % (int(self.season), int(self.episode)),
                '%02dx%02d' % (int(self.season), int(self.episode)),
                'season%02depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%02d' % (int(self.season), int(self.episode)),
                'season%depisode%d' % (int(self.season), int(self.episode))]

    def _label_settings(self):
        scraper_settings = json.loads(self.info['scraper_settings'])
        self.provider_color = settings.provider_color(self.scrape_provider)
        self.show_extra_info = scraper_settings['extra_info']
        self.show_filenames = scraper_settings['show_filenames']
        self.second_line_color = scraper_settings['multiline_highlight']
        self.highlight_type = scraper_settings['highlight_type']
        if self.highlight_type == '1':
            self.highlight_4K = scraper_settings['highlight_4K']
            self.highlight_1080p = scraper_settings['highlight_1080p']
            self.highlight_720p = scraper_settings['highlight_720p']
            self.highlight_SD = scraper_settings['highlight_SD']
        else:
            # Single Line...
            if self.provider_color == '':
                self.single_leading = ''
                self.single_closing = ''
            else:
                self.single_leading = '[COLOR=%s]' % self.provider_color
                self.single_closing = '[/COLOR]'
            # Multiline...
            if self.provider_color == '':
                self.multi1_leading = ''
                self.multi1_closing = ''
            else:
                self.multi1_leading = '[COLOR=%s]' % self.provider_color
                self.multi1_closing = '[/COLOR]'
            if self.second_line_color == '':
                self.multi2_leading = self.multi1_leading
                self.multi2_closing = self.multi1_closing
            else:
                self.multi2_leading = '[COLOR=%s]' % self.second_line_color
                self.multi2_closing = '[/COLOR]'

    def _build_label(self, file_name, details, size, video_quality):
        label, multiline_label1, multiline_label2 = build_internal_scrapers_label('PM CLOUD', self.show_filenames, self.show_extra_info, file_name, details, size, video_quality)
        if self.highlight_type == '1':
            if video_quality.upper() == '4K': leading_color = self.highlight_4K
            elif video_quality.upper()  == '1080P': leading_color = self.highlight_1080p
            elif video_quality.upper() == '720P': leading_color = self.highlight_720p
            else: leading_color = self.highlight_SD
            if self.second_line_color == '': multiline_open = leading_color
            else: multiline_open = self.second_line_color
            label = '[COLOR=%s]' % leading_color + label + '[/COLOR]'
            multiline_label = '[COLOR=%s]' % leading_color + multiline_label1 + '[/COLOR]' + '[COLOR=%s]' % multiline_open + multiline_label2 + '[/COLOR]'
        else:
            label = self.single_leading + label + self.single_closing
            multiline_label = self.multi1_leading + multiline_label1 + self.multi1_closing + self.multi2_leading + multiline_label2 + self.multi2_closing
        return label, multiline_label


