# -*- coding: utf-8 -*-
import xbmc
import json
from datetime import datetime, timedelta
try: from sqlite3 import dbapi2 as database
except ImportError: from pysqlite2 import dbapi2 as database
from apis.trakt_api import trakt_get_next_episodes
from windows.base_dialog import BaseDialog
from modules.indicators_bookmarks import detect_bookmark
from modules.settings import watched_indicators, check_database, default_all_episodes
from modules.nav_utils import translate_path, build_url
from modules.utils import local_string as ls
# from modules.utils import logger

class InfoPopupXML(BaseDialog):
	def __init__(self, *args, **kwargs):
		BaseDialog.__init__(self, args)
		self.selected = 'Exit'
		self.db_type = kwargs.get('db_type')
		self.extras_infolabel = xbmc.getInfoLabel('ListItem.Property(fen_extras_menu_params)')
		self.options_infolabel = xbmc.getInfoLabel('ListItem.Property(fen_options_menu_params)')
		self.get_meta()
		self.make_items()
		self.set_properties()

	def run(self):
		self.doModal()
		self.execute()

	def execute(self):
		if self.selected == 'Exit': return
		return self.execute_code(self.selected[0] % build_url(self.selected[1]))

	def onAction(self, action):
		if action in self.closing_actions:
			self.close()

	def onClick(self, controlID):
		self.get_extras_params()
		self.get_options_params()
		self.selected = {
						10: {'movie': ('RunPlugin(%s)', {'mode': 'play_media', 'vid_type': 'movie', 'query': self.meta['rootname'],
														'tmdb_id': self.meta['tmdb_id'], 'imdb_id': self.meta['imdb_id']}),
							'tvshow': ('ActivateWindow(Videos,%s,return)', self.make_tvshow_browse_params())},
						11: {'movie': ('RunPlugin(%s)', self.extras_params),
							'tvshow': ('RunPlugin(%s)', self.extras_params)},
						12: {'movie': ('RunPlugin(%s)', self.options_params),
							'tvshow': ('RunPlugin(%s)', self.options_params)},
						13: {'movie': ('RunPlugin(%s)', {'mode': 'trakt_manager_choice', 'tmdb_id': self.meta['tmdb_id'], 'imdb_id': self.meta['imdb_id'],
										'tvdb_id': 'None', 'db_type': 'movie'}),
							'tvshow': ('RunPlugin(%s)', {'mode': 'trakt_manager_choice', 'tmdb_id': self.meta['tmdb_id'], 'imdb_id': self.meta['imdb_id'],
										'tvdb_id': self.meta['tvdb_id'], 'db_type': 'tvshow'})},
						14: {'movie': 'Exit',
							'tvshow': 'Exit'}
						}[controlID][self.db_type]
		self.close()

	def make_items(self):
		if not self.meta: self.close()
		self.title = self.meta['title']
		self.year = str(self.meta['year'])
		self.poster = self.meta['poster']
		self.fanart = self.meta['fanart']
		self.clearlogo = self.meta['clearlogo']
		self.plot = self.meta['plot']
		self.rating = str(self.meta['rating'])
		self.mpaa = self.meta['mpaa']
		self.status = self.meta['extra_info'].get('status', '')
		self.sub_heading = self.meta['tagline'] if self.db_type == 'movie' and self.meta['tagline'] != '' else self.meta['genre']
		self.duration = self.get_duration()
		if self.db_type == 'movie':
			self.progress = self.get_progress()
			self.finish_watching = self.get_finish()
			self.last_aired_episode, self.next_aired_episode, self.next_episode = '', '', ''
		else:
			self.progress, self.finish_watching = '', ''
			self.last_aired_episode = self.get_last_aired()
			if self.status.lower() not in ('', 'ended', 'canceled'): self.next_aired_episode = self.get_next_aired()
			else: self.next_aired_episode = ''
			self.next_episode = self.get_next_episode()

	def set_properties(self):
		self.setProperty('tikiskins.db_type', self.db_type)
		self.setProperty('tikiskins.title', self.title)
		self.setProperty('tikiskins.year', self.year)
		self.setProperty('tikiskins.poster', self.poster)
		self.setProperty('tikiskins.fanart', self.fanart)
		self.setProperty('tikiskins.clearlogo', self.clearlogo)
		self.setProperty('tikiskins.plot', self.plot)
		self.setProperty('tikiskins.rating', self.rating)
		self.setProperty('tikiskins.mpaa', self.mpaa)
		self.setProperty('tikiskins.status', self.status)
		self.setProperty('tikiskins.sub_heading', self.sub_heading)
		self.setProperty('tikiskins.duration', self.duration)
		self.setProperty('tikiskins.progress', self.progress)
		self.setProperty('tikiskins.finish_watching', self.finish_watching)
		self.setProperty('tikiskins.last_aired_episode', self.last_aired_episode)
		self.setProperty('tikiskins.next_aired_episode', self.next_aired_episode)
		self.setProperty('tikiskins.next_episode', self.next_episode)

	def get_meta(self):
		self.meta = json.loads(xbmc.getInfoLabel('ListItem.Property(fen_listitem_meta)'))

	def get_extras_params(self):
		self.extras_params = json.loads(self.extras_infolabel)

	def get_options_params(self):
		self.options_params = json.loads(self.options_infolabel)

	def get_finish(self):
		if self.percent_watched in ('0', '100'): self.finished = xbmc.getInfoLabel('ListItem.EndTime')
		else:
			kodi_clock = xbmc.getInfoLabel('System.Time')
			if any(i in kodi_clock for i in ['AM', 'PM']): _format = '%I:%M %p'
			else: _format = '%H:%M'
			current_time = datetime.now()
			remaining_time = ((100 - int(self.percent_watched))/100) * self.duration_data
			finish_time = current_time + timedelta(minutes=remaining_time)
			self.finished = finish_time.strftime(_format)
		return '[B]%s: %s[/B]' % (ls(33062), self.finished)

	def get_duration(self):
		self.duration_data = int(float(self.meta['duration'])/60)
		duration_str = ls(33058) % self.duration_data
		return duration_str

	def get_progress(self):
		try: resume_point, curr_time = detect_bookmark(self.db_type, self.meta['tmdb_id'])
		except: resume_point = 0
		if resume_point in (0, '0', 0.0, '0.0'):
			playcount = self.meta['playcount']
			if playcount == 1: self.percent_watched = '100'
			else: self.percent_watched = '0'
		else: self.percent_watched = str(int(float(resume_point)))
		progress_status = '%s%% %s' % (self.percent_watched, ls(32475))
		return progress_status

	def get_last_aired(self):
		extra_info = self.meta['extra_info']
		if 'last_episode_to_air' in extra_info:
			last_ep = extra_info['last_episode_to_air']
			last_aired = 'S%.2dE%.2d' % (last_ep['season_number'], last_ep['episode_number'])
		else: last_aired = ''
		return '[B]%s:[/B] %s' % (ls(32634), last_aired)

	def get_next_aired(self):
		extra_info = self.meta['extra_info']
		if 'next_episode_to_air' in extra_info:
			next_ep = extra_info['next_episode_to_air']
			next_aired = 'S%.2dE%.2d' % (next_ep['season_number'], next_ep['episode_number'])
		else: return ''
		return '[B]%s:[/B] %s' % (ls(32635), next_aired)

	def get_next_episode(self):
		if watched_indicators() in (1, 2): ep_list = trakt_get_next_episodes(include_hidden=True, hidden_full_info=True)
		else:
			watched_db = translate_path('special://profile/addon_data/plugin.video.fen/watched_status.db')
			seen = set()
			check_database(watched_db)
			dbcon = database.connect(watched_db)
			dbcur = dbcon.cursor()
			dbcur.execute('''SELECT media_id, season, episode, last_played FROM watched_status WHERE db_type=?''', ('episode',))
			rows = dbcur.fetchall()
			rows = sorted(rows, key = lambda x: (x[0], x[1], x[2]), reverse=True)
			ep_list = [{"tmdb_id": int(a), "season": int(b), "episode": int(c), "last_played": d} for a, b, c, d in rows if not (a in seen or seen.add(a))]
		try: info = [i for i in ep_list if i['tmdb_id'] == self.meta['tmdb_id']][0]
		except: return ''
		current_season = info['season']
		current_episode = info['episode']
		season_data = self.meta['season_data']
		curr_season_data = [i for i in season_data if i['season_number'] == current_season][0]
		season = current_season if current_episode < curr_season_data['episode_count'] else current_season + 1
		episode = current_episode + 1 if current_episode < curr_season_data['episode_count'] else 1
		try: info = [i for i in season_data if i['season_number'] == season][0]
		except: return ''
		if info['episode_count'] >= episode: next_episode = 'S%.2dE%.2d' % (season, episode)
		return '[B]%s: %s[/B]' % (ls(33041), next_episode)

	def make_tvshow_browse_params(self):
		if self.db_type == 'movie': return {}
		try:
			total_seasons = self.meta['total_seasons']
			all_episodes = default_all_episodes()
			show_all_episodes = True if all_episodes in (1, 2) else False
			meta_json = json.dumps(self.meta)
			if show_all_episodes:
				if all_episodes == 1 and total_seasons > 1: url_params = {'mode': 'build_season_list', 'meta': meta_json, 'tmdb_id': self.meta['tmdb_id']}
				else: url_params = {'mode': 'build_episode_list', 'meta': meta_json, 'tmdb_id': self.meta['tmdb_id'], 'season': 'all'}
			else: url_params = {'mode': 'build_season_list', 'meta': meta_json, 'tmdb_id': self.meta['tmdb_id']}
			return url_params
		except: return {}