# -*- coding: utf-8 -*-
import xbmc, xbmcgui, xbmcvfs
import os
import time
import datetime
import xml.etree.ElementTree as ET
from windows import open_window
from caches import check_databases, clean_databases
from modules.settings_reader import get_setting, set_setting, make_settings_dict
from modules.nav_utils import sync_MyAccounts
from modules.kodi_utils import translate_path, get_kodi_version, execute_builtin
from modules.utils import gen_file_hash, local_string, confirm_dialog, ok_dialog
from modules import settings
from modules.utils import logger

window = xbmcgui.Window(10000)
monitor = xbmc.Monitor()
addon_dir = translate_path('special://home/addons/plugin.video.fen')
db_types = ['movie', 'tvshow']

class InitializeDatabases:
	def run(self):
		logger('FEN', 'InitializeDatabases Service Starting')
		try: check_databases()
		except: pass
		return logger('FEN', 'InitializeDatabases Service Finished')

class CheckSettingsFile:
	def run(self):
		logger('FEN', 'CheckSettingsFile Service Starting')
		window.clearProperty('fen_settings')
		profile_dir = translate_path('special://profile/addon_data/plugin.video.fen/')
		if not xbmcvfs.exists(profile_dir):
			xbmcvfs.mkdirs(profile_dir)
		settings_xml = os.path.join(profile_dir, 'settings.xml')
		if not xbmcvfs.exists(settings_xml):
			__addon__ = settings.addon()
			addon_version = __addon__.getAddonInfo('version')
			__addon__.setSetting('version_number', addon_version)
			monitor.waitForAbort(0.5)
		make_settings_dict()
		return logger('FEN', 'CheckSettingsFile Service Finished')

class SyncMyAccounts:
	def run(self):
		logger('FEN', 'SyncMyAccounts Service Starting')
		sync_MyAccounts(silent=True)
		return logger('FEN', 'SyncMyAccounts Service Finished')

class ReuseLanguageInvokerCheck:
	def run(self):
		logger('FEN', 'ReuseLanguageInvokerCheck Service Starting')
		if get_kodi_version() < 18: return
		addon_xml = os.path.join(addon_dir, 'addon.xml')
		tree = ET.parse(addon_xml)
		root = tree.getroot()
		current_addon_setting = get_setting('reuse_language_invoker', 'true')
		try: current_xml_setting = [str(i.text) for i in root.iter('reuselanguageinvoker')][0]
		except: return logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
		if current_xml_setting == current_addon_setting:
			return logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
		for item in root.iter('reuselanguageinvoker'):
			item.text = current_addon_setting
			hash_start = gen_file_hash(addon_xml)
			tree.write(addon_xml)
			hash_end = gen_file_hash(addon_xml)
			logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
			if hash_start != hash_end:
				if not confirm_dialog(text='%s\n%s' % (local_string(33021), local_string(33020))): return
				current_profile = xbmc.getInfoLabel('system.profilename')
				execute_builtin('LoadProfile(%s)' % current_profile)
			else: ok_dialog(text=ls(32574), top_space=True)

class ClearSubs:
	def run(self):
		logger('FEN', 'Clear Subtitles Service Starting')
		subtitle_path = translate_path('special://temp/')
		files = xbmcvfs.listdir(subtitle_path)[1]
		for i in files:
			try:
				if i.startswith('FENSubs_'): xbmcvfs.delete(os.path.join(subtitle_path, i))
				if i.endswith('.nfo'): xbmcvfs.delete(os.path.join(subtitle_path, i))
			except: pass
		return logger('FEN', 'Clear Subtitles Service Finished')

class ClearTraktServices:
	def run(self):
		logger('FEN', 'Trakt Service Starting')
		if settings.refresh_trakt_on_startup():
			try:
				from caches.trakt_cache import clear_all_trakt_cache_data
				success = clear_all_trakt_cache_data(silent=True)
			except: pass
		return logger('FEN', 'Trakt Service Finished')

class DatabaseMaintenance:
	def run(self):
		monitor.waitForAbort(10.0)
		self.time = datetime.datetime.now()
		current_time = self._get_timestamp(self.time)
		due_clean = int(get_setting('database.maintenance.due', '0'))
		if current_time > due_clean:
			logger('FEN', 'Database Maintenance Service Starting')
			next_clean = str(int(self._get_timestamp(self.time + datetime.timedelta(days=7))))
			try: clean_databases(current_time, database_check=False, silent=True)
			except: pass
			set_setting('database.maintenance.due', next_clean)
			return logger('FEN', 'Database Maintenance Service Finished')

	def _get_timestamp(self, date_time):
		return int(time.mktime(date_time.timetuple()))

class ListItemPopups():
	def run(self):
		logger('FEN', 'Listitem Monitor Service Starting')
		previous_label, current_label, highlight_time = '', '', 0
		self.delay = 2.0
		self.skin_location = settings.skin_location()
		while not monitor.abortRequested():
			if get_setting('popup.enabled', 'false') == 'true':
				try:
					previous_label, highlight_time, activate_function, current_dbtype, current_label = self.getInfo(previous_label, highlight_time)
					if activate_function:
						try: self.processInfoPopUp(current_dbtype, current_label)
						except: pass
				except: pass
				monitor.waitForAbort(self.delay)
			else:
				monitor.waitForAbort(10.0)
		return logger('FEN', 'Listitem Monitor Service Finished')

	def getInfo(self, previous_label, highlight_time):
		activate_function = False
		in_fen = xbmc.getInfoLabel('Container.PluginName') == 'plugin.video.fen'
		widget = xbmc.getInfoLabel('ListItem.Property(fen_widget)') == 'true'
		current_dbtype = xbmc.getInfoLabel('ListItem.dbtype')
		current_label = xbmc.getInfoLabel('ListItem.label')
		if current_label == '': return previous_label, highlight_time, activate_function, current_dbtype, current_label
		proceed = (in_fen or widget) and current_dbtype in db_types
		if proceed:
			self.delay = 0.1
			if current_label != previous_label:
				highlight_time = time.time()
			pause = time.time() - highlight_time
			previous_label = current_label
			try: delay = float(int(get_setting('popup.delay')))/1000
			except: delay = float(3500)/1000
			if highlight_time and pause >= delay:
				activate_function = True
				highlight_time = 0
		else:
			self.delay = 2.0
			previous_label = ''
		return previous_label, highlight_time, activate_function, current_dbtype, current_label

	def processInfoPopUp(self, current_dbtype, current_label):
		if self.visibilityCheck(current_dbtype, current_label):
			open_window(('windows.infopopup', 'InfoPopup'), 'infopopup.xml', self.skin_location, db_type=current_dbtype)

	def visibilityCheck(self, current_dbtype, current_label):
		if not xbmc.getInfoLabel('ListItem.dbtype') == current_dbtype: return False
		if not xbmc.getInfoLabel('ListItem.label') == current_label: return False
		current_id = xbmcgui.getCurrentWindowId()
		if not current_id in (10000, 10025): return False
		if not xbmc.getCondVisibility('Control.HasFocus(%s)' % str(xbmcgui.Window(current_id).getFocusId())): return False
		if monitor.abortRequested(): return False
		if xbmc.Player().isPlaying(): return False
		return True
