# -*- coding: utf-8 -*-
import xbmc, xbmcgui, xbmcvfs
import os
import time
import datetime
import xml.etree.ElementTree as ET
from windows import open_window
from caches import check_databases, clean_databases
from modules.view_utils import set_view_properties
from modules.settings_reader import get_setting, set_setting, make_settings_dict
from modules.nav_utils import sync_MyAccounts
from modules.kodi_utils import translate_path, get_kodi_version, execute_builtin, confirm_dialog, ok_dialog, local_string
from modules.utils import gen_file_hash
from modules import settings
from modules.utils import logger

window = xbmcgui.Window(10000)
monitor = xbmc.Monitor()
addon_dir = translate_path('special://home/addons/plugin.video.fen')
db_types = ['movie', 'tvshow']

class InitializeDatabases:
	def run(self):
		logger('FEN', 'InitializeDatabases Service Starting')
		try: check_databases()
		except: pass
		return logger('FEN', 'InitializeDatabases Service Finished')

class CheckSettingsFile:
	def run(self):
		logger('FEN', 'CheckSettingsFile Service Starting')
		window.clearProperty('fen_settings')
		profile_dir = translate_path('special://profile/addon_data/plugin.video.fen/')
		if not xbmcvfs.exists(profile_dir):
			xbmcvfs.mkdirs(profile_dir)
		settings_xml = os.path.join(profile_dir, 'settings.xml')
		if not xbmcvfs.exists(settings_xml):
			__addon__ = settings.addon()
			addon_version = __addon__.getAddonInfo('version')
			__addon__.setSetting('version_number', addon_version)
			monitor.waitForAbort(0.5)
		make_settings_dict()
		return logger('FEN', 'CheckSettingsFile Service Finished')

class SyncMyAccounts:
	def run(self):
		logger('FEN', 'SyncMyAccounts Service Starting')
		sync_MyAccounts(silent=True)
		return logger('FEN', 'SyncMyAccounts Service Finished')

class ClearSubs:
	def run(self):
		logger('FEN', 'Clear Subtitles Service Starting')
		subtitle_path = translate_path('special://temp/')
		files = xbmcvfs.listdir(subtitle_path)[1]
		for i in files:
			try:
				if i.startswith('FENSubs_'): xbmcvfs.delete(os.path.join(subtitle_path, i))
				if i.endswith('.nfo'): xbmcvfs.delete(os.path.join(subtitle_path, i))
			except: pass
		return logger('FEN', 'Clear Subtitles Service Finished')

class ClearTraktServices:
	def run(self):
		logger('FEN', 'Trakt Service Starting')
		if settings.refresh_trakt_on_startup():
			try:
				from caches.trakt_cache import clear_all_trakt_cache_data
				success = clear_all_trakt_cache_data(silent=True)
			except: pass
		return logger('FEN', 'Trakt Service Finished')

class ReuseLanguageInvokerCheck:
	def run(self):
		logger('FEN', 'ReuseLanguageInvokerCheck Service Starting')
		if get_kodi_version() < 18: return
		addon_xml = os.path.join(addon_dir, 'addon.xml')
		tree = ET.parse(addon_xml)
		root = tree.getroot()
		current_addon_setting = get_setting('reuse_language_invoker', 'true')
		try: current_xml_setting = [str(i.text) for i in root.iter('reuselanguageinvoker')][0]
		except: return logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
		if current_xml_setting == current_addon_setting:
			return logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
		for item in root.iter('reuselanguageinvoker'):
			item.text = current_addon_setting
			hash_start = gen_file_hash(addon_xml)
			tree.write(addon_xml)
			hash_end = gen_file_hash(addon_xml)
			logger('FEN', 'ReuseLanguageInvokerCheck Service Finished')
			if hash_start != hash_end:
				if not confirm_dialog(text='%s\n%s' % (local_string(33021), local_string(33020))): return
				current_profile = xbmc.getInfoLabel('system.profilename')
				execute_builtin('LoadProfile(%s)' % current_profile)
			else: ok_dialog(text=32574, top_space=True)

class ViewsSetWindowProperties:
	def run(self):
		logger('FEN', 'ViewsSetWindowProperties Service Starting')
		try: set_view_properties()
		except: pass
		return logger('FEN', 'ViewsSetWindowProperties Service Finished')

class ShowChangelog:
	def run(self):
		logger('FEN', 'ShowChangelog Service Starting')
		addon_version = settings.addon().getAddonInfo('version')
		setting_version = get_setting('version_number')
		if addon_version != setting_version:
			set_setting('version_number', addon_version)
			window.setProperty('fen_display_changelog', 'true')
		else: window.setProperty('fen_display_changelog', 'false')
		return logger('FEN', 'ShowChangelog Service Finished')

class AutoRun:
	def run(self):
		try:
			logger('FEN', 'AutoRun Service Starting')
			if settings.auto_start_fen(): execute_builtin('RunAddon(plugin.video.fen)')
			logger('FEN', 'AutoRun Service Finished')
			return
		except: return

class DatabaseMaintenance:
	def run(self):
		self.time = datetime.datetime.now()
		current_time = self._get_timestamp(self.time)
		due_clean = int(get_setting('database.maintenance.due', '0'))
		if current_time >= due_clean:
			logger('FEN', 'Database Maintenance Service Starting')
			monitor.waitForAbort(10.0)
			try: clean_databases(current_time, database_check=False, silent=True)
			except: pass
			next_clean = str(int(self._get_timestamp(self.time + datetime.timedelta(days=3))))
			set_setting('database.maintenance.due', next_clean)
			return logger('FEN', 'Database Maintenance Service Finished')

	def _get_timestamp(self, date_time):
		return int(time.mktime(date_time.timetuple()))
